package mcfall.raytracer.objects;

import java.util.HashMap;
import java.util.List;

import org.apache.log4j.or.ObjectRenderer;

import mcfall.math.IncompatibleMatrixException;
import mcfall.math.Matrix;
import mcfall.math.NotInvertibleException;
import mcfall.math.Point;
import mcfall.math.Ray;
import mcfall.math.SquareMatrix;
import mcfall.math.Vector;
import mcfall.raytracer.Intensity;
import mcfall.raytracer.Material;
import mcfall.raytracer.ThreeDimensionalObject;

public abstract class MathematicalObject implements ThreeDimensionalObject, ObjectRenderer {
	
	/**
	 * Maintains a mapping between object types and the next auto-generated ID associated
	 * with that type
	 */
	private static HashMap<String, Integer> objectIDMap;
	
	/**
	 * The name associated with this object, used to fulfill the ThreeDimensionalObject method getName()
	 */
	private String name;
	
	/**
	 * A matrix that specifies the current transformation matrix
	 */
	protected Matrix currentTransformation;
	
	/**
	 * The inverse of the current transformation matrix
	 */
	protected Matrix inverseTransform;
	
	/**
	 * The transpose of the current transformation's inverse, used for computing normal vectors at hit locations
	 */
	protected Matrix inverseTranspose;
	
	/**
	 * The material that this object is made of
	 */
	private Material material;
	
	/**
	 * A value between 0 and 1 that describes how reflective this object 
	 * is.  0 indicates not reflective at all, while 1 indicates complete
	 * reflectivity
	 */
	private Intensity reflectivity;
	
	private Intensity refractionIndex;
	
	private Intensity transparency;
	/**
	 * Determines the time(s) at which this object is hit by the ray <i>ray</i>
	 * @param ray
	 * @return an array of double values, which can be either positive (in which case the ray actually hits the object)
	 * or negative (in which case the object is "behind" the origin of the ray, and is therefore not hit)
	 */
	protected abstract List<HitRecord> genericHitTime (Ray ray);
			
	/**
	 * Gets the type of this mathematical object
	 * @return a string uniquely identifying the type of this object
	 */
	protected abstract String getObjectType ();
	
	/**
	 * Initialize the static data associated with this class
	 */
	static {
		objectIDMap = new HashMap<String,Integer> ();
	}
	
	/**
	 * Constructs a MathematicalObject with an automatically generated name, which is of the form
	 * ObjectType-ID, where ID is a unique number identifying the object
	 * @see getObjectType
	 *
	 */
	protected MathematicalObject () {
		currentTransformation = SquareMatrix.createIdentityMatrix(4);
		inverseTransform = currentTransformation;
		inverseTranspose = inverseTransform.transpose();
		reflectivity = new Intensity(0.0);
		refractionIndex = new Intensity(0.0);
		transparency = new Intensity(0.0);
		String objectType = getObjectType ();
		
		/**
		 * Make sure this method is synchronized so that just in case, we don't run into problems with concurrent access
		 */
		Integer idValue = null;
		synchronized (MathematicalObject.class) {
			/*  Look up the current ID value in the objectIDMap  */
			idValue = objectIDMap.get(objectType);
			int newID = 1;
			
			/*  Increment existing value by 1 if it exists, otherwise put a value of 1 into the map  */
			if (idValue != null) {
				idValue = new Integer (idValue.intValue()+1);
			}
			else {
				idValue = new Integer (newID);				
			}
			
			/*  Replace the existing value in objectIDMap with the value just used  */
			objectIDMap.put(objectType, idValue);
		}
		this.name = objectType + "-" + idValue.intValue(); 
	}
	
	/**
	 * Constructs a MathematicalObject with the given name, and initializes the current transformation 
	 * to be the identity transformation
	 * @param name the name to assign to the object
	 */
	protected MathematicalObject (String name) {		
		currentTransformation = SquareMatrix.createIdentityMatrix(4);
		inverseTransform = currentTransformation;
		inverseTranspose = inverseTransform.transpose();
		reflectivity = new Intensity(0.0);
		this.name = name; 
	}
	
	/**
	 * Applies the specified transformation by postmultiplying the current transformation with it
	 * @param transform the transformation to be added into the current transformation
	 */
	public void transform(Matrix transform) {
		try {
			currentTransformation = currentTransformation.postmultiply(transform);
			inverseTransform = currentTransformation.invert();
			inverseTranspose = inverseTransform.transpose();
		}
		catch (IncompatibleMatrixException incompatible) {
			throw new RuntimeException (incompatible);
		} catch (NotInvertibleException e) {
			throw new RuntimeException (e);
		}
	}

	/**
	 * Computes the time(s) at which the ray <i>ray</i> hits this object, if any
	 * @return a java.util.List of HitRecord objects, with one record for each hit.  If the object
	 * is not hit by <i>ray</i>, then an empty list is returned  
	 */
	public List<HitRecord> hitTime(Ray ray) {		
		return genericHitTime(ray.transform(inverseTransform));						
	}
	
	protected Vector getTransformedVector (Vector genericNormalVector) throws IncompatibleMatrixException {
		return Vector.fromColumnMatrix(inverseTranspose.postmultiply(genericNormalVector));
	}
	
	public String getName () {
		return name;
	}
	
	public String doRender (Object o) {
		MathematicalObject object = (MathematicalObject) o;
		return object.getName ();
	}

	public Material getMaterial() {
		return material;
	}

	public void setMaterial(Material material) {
		this.material = material;
	}
		
	public Matrix getTransform () {
		return currentTransformation;
	}

	
	public double getReflectionCoefficient() {
		return reflectivity.getValue();
	}

	public void setReflectionCoefficient (double reflectivity) {
		this.reflectivity = new Intensity(reflectivity);
	}
	public double getRefractionIndex() {
		if(refractionIndex==null) {
			refractionIndex = new Intensity(0.0);
		}
		return refractionIndex.getValue();
	}
	public void setRefractionIndex(double refractionPercent) {
		this.refractionIndex  = new Intensity(refractionPercent);
	}
	public double getTransparency()
	{
		return transparency.getValue();
	}
	public void setTransparency(double transparency) 
	{
		this.transparency = new Intensity(transparency);
	}
}
